﻿using System;
using System.Reflection;
using System.Linq;
using System.Collections.Generic;

using AutoMapper;

namespace HIPS.Web.UI.Helpers.Mapping
{

    /// <summary>
    /// Provides mapping between object structures.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="18 November 2013">Initial version.</change>
    /// </history>
    internal class ObjectMapper
    {

        #region Constructors

        /// <summary>
        /// Static constructor.
        /// </summary>
        /// <remarks>Used to initialise AutoMapper mapping profiles.</remarks>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 November 2013">Initial version.</change>
        /// </history>
        static ObjectMapper()
        {

            Mapper.Initialize(cfg =>
            {
                var profileType = typeof(Profile);
                // Get an instance of each Profile with a parameterless constructor in the executing assembly.
                var profiles = Assembly.GetExecutingAssembly().GetTypes().Where(t => profileType.IsAssignableFrom(t) && t.GetConstructor(Type.EmptyTypes) != null)
                    .Select(Activator.CreateInstance)
                    .Cast<Profile>();

                // Initialize AutoMapper with each instance of the profiles found.
                profiles.ToList().ForEach(p => cfg.AddProfile(p));

                //Add any other profiles that do not have a parameterless constructor here...
                //...
            });

        }

        #endregion

        #region Methods

        /// <summary>
        /// Execute a mapping from the source object to a new destination object. The source type is inferred from the source object.
        /// </summary>
        /// <typeparam name="TDestination">Type parameter representing the destination type instance to be created by the mapping.</typeparam>
        /// <param name="source">Source instance to be used as the source for the mapping.</param>
        /// <returns>An instance of TDestination created by executing a mapping from the source type derived from the source instance to the destination type specified by TDestination.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 November 2013">Initial version.</change>
        /// </history>
        public static TDestination Map<TDestination>(object source)
        {
            return Mapper.Map<TDestination>(source);
        }

        /// <summary>
        /// Execute a mapping from the source object to a new destination object. The source type is inferred from the source object.
        /// </summary>
        /// <typeparam name="TDestination">Type parameter representing the destination type instance to be created by the mapping.</typeparam>
        /// <param name="source">Source instance to be used as the source for the mapping.</param>
        /// <param name="context">Object containing context to be used during mapping.</param>
        /// <returns>An instance of TDestination created by executing a mapping from the source type derived from the source instance to the destination type specified by TDestination.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
        /// </history>
        public static TDestination Map<TDestination>(object source, object context)
        {
            return Mapper.Map<TDestination>(source, delegate(IMappingOperationOptions m) { m.Items.Add("context", context); });
        }

        /// <summary>
        /// Execute a mapping from the source object to a new destination object. The source type is inferred from the source object.
        /// </summary>
        /// <typeparam name="TDestination">Type parameter representing the destination type instance to be created by the mapping.</typeparam>
        /// <param name="source">Source instance to be used as the source for the mapping.</param>
        /// <param name="mapParameters">IDictionary containing parameters to be used within the map.</param>
        /// <returns>An instance of TDestination created by executing a mapping from the source type derived from the source instance to the destination type specified by TDestination.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 February 2014">Initial version.</change>
        /// </history>
        public static TDestination Map<TDestination>(object source, IDictionary<string, object> mapParameters)
        {
            return Mapper.Map<TDestination>(source, delegate(IMappingOperationOptions m) { mapParameters.ToList().ForEach(p => m.Items.Add(p.Key, p.Value)); });
        }

        #endregion

    }

}
